
#include "..\config.h"
#include "string.h"
#include "userDataProcess.h"

// Delay ms
// input: 0~65535 
void delayUs(INT16U us) {
    INT16U i=0;
	INT16U j=0;
    for(i=0; i<=us; i++) {
	    for(j=0; j<=16; j++);
	}
}

// Delay us
// input: 0~65535 
void delayMs(INT16U ms) {
    INT16U i=0;
    for(i=0; i<=ms; i++) {
	    delayUs(1000);
	}
}

// Initialize Key1 ~ Key4 PIN as input
void keyInit(void) {
    FIO2DIR &= ~KEY1;        // Set P2.10(KEY1) as input
	FIO2DIR &= ~KEY2;        // Set P2.11(KEY2) as input
    FIO2DIR &= ~KEY3;        // Set P2.12(KEY3) as input
    FIO2DIR &= ~KEY4;        // Set P2.13(KEY4) as input
}

// Set DB7~DB0 PIN as input or ourput
// direction=1 -> input   direction=0(or any other value non-1) -> output;   
void lcdDdataPinsDirectionSet(INT8U direction) {
    if(direction==1) {
		FIO0DIR2 = 0x00;      // Set P0.23~P0.16 as input
	}
	else {
		FIO0DIR2 = 0xFF;      // Set P0.23~P0.16 as output
	}
}

// Initialize LCD control PINs as output; RST, CD, RW, E;
void lcdPinInit(void) {
    FIO0DIR |= RST;                 // set P0.15(RST) as output
	FIO1DIR |= CD;                  // set P1.31(A0) as output
	FIO3DIR |= RW;                  // set P3.25(RW) as output
	FIO3DIR |= E;                   // set P3.26(E) as output

	lcdDdataPinsDirectionSet(0);    // set data pins as output
}

// Set RST level
// Input:   0->Low;  1->High
void setRST(INT8U RST_value) {
	if(RST_value == 0) {
		FIO0CLR = RST;	     // RST = 0
	}
	else {
		FIO0SET = RST;		 // RST = 1
	}
}  

// Set CD level
// Input:   0->Low;  1->High
void setCD(INT8U CD_value) {
	if(CD_value == 0) {
		FIO1CLR = CD;	     // CD = 0
	}
	else {
		FIO1SET = CD;		 // CD = 1
	}
}

// Set RW level
// Input:   0->Low;  1->High
void setRW(INT8U RW_value) {
	if(RW_value == 0) {
		FIO3CLR = RW;	     // RW = 0
	}
	else {
		FIO3SET = RW;		 // RW = 1
   }
}

// Set E level
// Input:   0->Low;  1->High
void setE(INT8U E_value) {
	if(E_value == 0) {
		FIO3CLR = E;	     // E = 0
	}
	else {
		FIO3SET = E;		 // E = 1
    }
}



// Write command
// Input:  Refer to Command Set	of Data Sheet
void writeComd(INT8U comd) {

    lcdDdataPinsDirectionSet(0);	      // DB7~DB0 as output
	
	setRW(0);  setCD(0);				  // Write instruction register

	FIO0PIN2 = comd;  // output command on P0.23~P0.16, as a 8bits bus
    delayUs(10);

	setE(1);   delayUs(5);
	setE(0);   delayUs(10);
}



// Wirte data,
// Input:  Refer to Data Sheet
void writeData(INT8U data) {
	
	lcdDdataPinsDirectionSet(0);	      // DB7~DB0 as output

	setRW(0);  setCD(1);				  // Write data register

	FIO0PIN2 = data;  // output data on P0.23~P0.16, as a 8bits bus
    delayUs(10);

	setE(1);   delayUs(5);
	setE(0);   delayUs(10);
}

// Initialize LCM controller IC, ST7565P
void lcdInit(void) {

    setRST(0);
    delayMs(10);
	setRST(1);
	delayMs(10);		 // Hard reset ST7565

	writeComd(0xC8);    delayUs(100);   // "Common Output Mode Select": COM63->COM0
	writeComd(0x26);    delayUs(100);	// "VO, Voltage Regulator Internal Resistor Ratio Set"   111, the largest
	writeComd(0x81);    delayUs(100);	// "Electronic Volume Register Set"_(1) 
	writeComd(0x22);    delayUs(100);	// "Electronic Volume Register Set"_(2)	  100010

	writeComd(0x2F);    delayMs(100);	// "Power Controller Set", Booster ON, Voltage regulator ON, Voltage follower ON

	writeComd(0x40);	delayUs(100);	// "Display start line set"

	writeComd(0xA4);	delayUs(100);	// "Display all points ON/OFF", normal display
	writeComd(0xAF);	delayUs(100);	// "Display ON/OFF"	  ON

}

// **** fillScreen ****
// left -> right, up -> down, to fill screen
// this function just fill screen with simple data
void fillScreen(INT8U data) {
    INT16U i,j;
    for(j=0;j<8;j++) {
        writeComd(0xb0+j);   //set page address D0~D7
        writeComd(0x40);     //set start line
        writeComd(0x10);     //set column Address
        writeComd(0x00);
 
        for(i=0;i<132;i++){
            writeData(data);
        }
    } 
}

// **** cleanScreen ****
void cleanScreen(void)	{
    fillScreen(0x00);
}

// **** placeDot ****
// Place a dot at (x,y) on screen. (0,0) at the up left corner
// input:  x position (0~127)
//         y position (0~63)
// output: void
void placeDot(INT16U x, INT16U y) {
    if ( (x<=127) && (y<=63) ) {	    // protect overflow
        writeComd(0x10|(x>>4));	        // set x  (1)
	    writeComd(0x00|(x&0x0f));		// set x  (2) 
		writeComd(0xb0|(y/8));		    // set y; actually set current page
		writeData(1<<(y%8));            // write converted data
    }
}

void refreshScreen(INT8U *gData) {
    INT16U i,j;
    for(j=0;j<8;j++) {
        writeComd(0xb0+j);   //set page address D0~D7

        writeComd(0x10);     //set column Address
        writeComd(0x00);
 
        for(i=0;i<128;i++){
            writeData(gData[i+(j*128)]);
        }

    } 
}


INT8U graphic_D0[1024]=	   // OD logo & PN & resolution	 for "AMG12864PR-G-B12WTDW"
{
0x00,0x00,0x00,0x00,0xC0,0x60,0x30,0x18,0x18,0x3C,0xE4,0xE4,0x34,0x34,0x34,0x34,
0x2C,0x2C,0x58,0xD8,0xB0,0x60,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0xC0,
0x40,0xC0,0xC0,0x00,0x00,0xC0,0xC0,0xC0,0x40,0xC0,0xC0,0x00,0x00,0x80,0xC0,0xC0,
0x00,0x80,0xC0,0xC0,0x40,0x40,0x40,0x00,0xC0,0xC0,0xC0,0xC0,0x00,0x00,0xC0,0xC0,
0x00,0x40,0xC0,0xC0,0xC0,0xC0,0x40,0x40,0x00,0x00,0x00,0x00,0xC0,0xC0,0xC0,0xC0,
0xC0,0xC0,0x00,0x00,0xC0,0xC0,0x40,0x00,0xC0,0xC0,0x40,0xC0,0xC0,0xC0,0x00,0x00,
0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0x00,0x00,0xC0,0xC0,0x40,0x00,0x00,0x00,0x00,0x00,
0x00,0xC0,0xC0,0xC0,0x00,0xC0,0xC0,0xC0,0x00,0x00,0x00,0xC0,0xC0,0x00,0x00,0x00,
0x00,0x00,0x38,0xFF,0x01,0x00,0x00,0x00,0x00,0xFC,0xFF,0x01,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0xF1,0xFF,0x1E,0x83,0xFF,0x00,0x00,0x00,0x80,0xFC,0xFF,0x0F,0x00,
0xF8,0xFF,0x1F,0x00,0xF8,0xFF,0x3F,0x10,0xF8,0xFF,0x07,0x00,0xE0,0xFF,0x7F,0x01,
0xE0,0xFF,0x7F,0x11,0x10,0x10,0x00,0xF0,0xFF,0x1F,0xFF,0xFF,0xF0,0xFC,0x3F,0x01,
0x00,0x80,0xFE,0xFF,0x0F,0x00,0x00,0x00,0x00,0x00,0xC0,0xFF,0xFF,0x0F,0x80,0xFE,
0xFF,0x1F,0xC0,0xFF,0xFF,0x0F,0x80,0x8F,0x9F,0x3F,0xF8,0xF2,0xC3,0x03,0xC0,0xFE,
0xFF,0x67,0x60,0x7C,0x3F,0x1F,0xC0,0xFE,0xFF,0x0F,0x00,0x00,0x00,0x80,0xF0,0xFC,
0x9F,0x8F,0xFF,0xFF,0x00,0x00,0xFF,0xFF,0xF8,0x3C,0x0F,0x03,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x01,0x07,0x0C,0x18,0x30,0x3E,0x3F,0x78,0x58,0x58,0x58,0x58,0x48,
0x4C,0x26,0x27,0x13,0x18,0x0E,0x03,0x01,0x00,0x00,0x00,0x0F,0x0F,0x09,0x08,0x0E,
0x0F,0x03,0x00,0x0E,0x0F,0x03,0x00,0x0E,0x0F,0x0F,0x00,0x08,0x0F,0x0F,0x00,0x08,
0x0F,0x0F,0x08,0x08,0x08,0x00,0x0C,0x0F,0x00,0x00,0x01,0x0F,0x0F,0x0F,0x00,0x00,
0x00,0x0F,0x0F,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x0F,0x0D,0x0C,0x0F,0x07,
0x03,0x08,0x0F,0x0F,0x03,0x00,0x0F,0x0F,0x0C,0x0C,0x0F,0x0F,0x03,0x08,0x0F,0x0F,
0x01,0x00,0x00,0x00,0x00,0x00,0x0F,0x0F,0x09,0x08,0x08,0x00,0x0C,0x0F,0x0F,0x01,
0x00,0x01,0x0F,0x0F,0x00,0x08,0x0F,0x0F,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0xC0,0x00,0x00,0x00,0xC0,0xC0,0x00,0x00,0x00,0xC0,0xC0,0x00,
0x80,0x40,0x40,0x40,0x40,0x80,0x00,0x00,0x80,0xC0,0x00,0x00,0x80,0x40,0x40,0x80,
0x00,0x80,0x40,0x40,0x80,0x00,0x80,0x40,0x40,0x80,0x00,0x00,0x00,0x80,0xC0,0x00,
0xC0,0x40,0x40,0x40,0x80,0x00,0xC0,0x40,0x40,0x40,0x40,0x80,0x00,0x00,0x00,0x00,
0x00,0x80,0x40,0x40,0x40,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0x40,
0x40,0x40,0x80,0x00,0x00,0x80,0xC0,0x00,0x00,0x80,0x40,0x40,0x80,0x00,0xC0,0x00,
0x00,0x00,0xC0,0x00,0x00,0x00,0xC0,0x00,0x40,0x40,0xC0,0x40,0x40,0x00,0xC0,0x40,
0x40,0x40,0x40,0x80,0x00,0xC0,0x00,0x00,0x00,0xC0,0x00,0x00,0x00,0xC0,0x00,0x00,
0x00,0x00,0x18,0x07,0x04,0x07,0x18,0x00,0x1F,0x00,0x07,0x18,0x07,0x00,0x1F,0x00,
0x0F,0x10,0x10,0x12,0x12,0x0E,0x00,0x01,0x00,0x1F,0x00,0x00,0x18,0x14,0x12,0x11,
0x00,0x0D,0x12,0x12,0x0D,0x00,0x0F,0x12,0x12,0x0C,0x00,0x06,0x05,0x04,0x1F,0x04,
0x1F,0x02,0x02,0x02,0x01,0x00,0x1F,0x02,0x02,0x02,0x06,0x09,0x10,0x04,0x04,0x04,
0x00,0x0F,0x10,0x10,0x12,0x12,0x0E,0x00,0x04,0x04,0x04,0x00,0x00,0x00,0x1F,0x12,
0x12,0x12,0x0D,0x00,0x00,0x00,0x1F,0x00,0x00,0x18,0x14,0x12,0x11,0x00,0x00,0x0F,
0x10,0x0F,0x00,0x0F,0x10,0x0F,0x00,0x00,0x00,0x00,0x1F,0x00,0x00,0x00,0x1F,0x10,
0x10,0x10,0x10,0x0F,0x00,0x00,0x0F,0x10,0x0F,0x00,0x0F,0x10,0x0F,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0x00,0x00,0x00,
0x80,0x40,0x40,0x40,0x80,0x00,0x80,0x40,0x40,0x40,0x80,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x40,0x40,0x80,0x00,0x00,0x00,0x80,0xC0,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x3F,0x00,0x00,0x00,
0x20,0x30,0x28,0x24,0x23,0x00,0x1D,0x22,0x22,0x22,0x1D,0x00,0x00,0x00,0x00,0x0A,
0x07,0x0A,0x00,0x00,0x00,0x00,0x1F,0x22,0x22,0x22,0x1C,0x00,0x0C,0x0B,0x08,0x3F,
0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D1[1024]=	   // Flower
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x0C,0x04,
0x08,0x08,0x10,0x30,0x40,0x80,0x00,0x80,0xFC,0x06,0x01,0x01,0x01,0x06,0x78,0xC0,
0x00,0x00,0x00,0x00,0xC0,0x60,0x10,0x08,0x04,0x04,0x04,0x04,0x0C,0xF8,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x80,0x80,0x80,0x80,0x80,0x80,0xBE,0xC2,0x02,0x04,0x04,0x04,0x08,0x19,0x36,0x78,
0xE0,0x80,0x00,0x00,0x00,0x01,0x03,0x0F,0x78,0xC0,0x00,0x00,0x00,0x00,0x00,0xFF,
0xC0,0x30,0x0C,0x03,0x00,0x00,0x00,0x00,0x00,0x80,0x60,0x30,0x1E,0x09,0x04,0x06,
0x02,0x02,0x01,0x81,0x61,0x39,0x86,0x80,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x07,0x0C,0x10,0x20,0x40,0x40,0x80,0x81,0x01,0x06,0x0C,0x18,0x30,0x60,0xC0,0x80,
0x00,0x01,0x03,0x06,0x0C,0x30,0x60,0x80,0x00,0x03,0x1E,0xF0,0x00,0x00,0xF8,0x1F,
0x01,0x00,0x00,0x80,0x60,0x30,0x08,0x04,0x03,0x01,0x00,0x00,0x80,0xC0,0x60,0x30,
0x18,0x04,0x03,0x03,0x83,0x81,0xC1,0x40,0x60,0x50,0x58,0x4F,0x81,0x80,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x0E,0x12,
0x21,0x61,0x41,0x41,0x81,0x81,0x01,0x01,0x01,0x01,0x03,0x03,0x03,0x82,0x82,0x86,
0x85,0x8D,0x8A,0x9E,0x9C,0xB8,0xB8,0xF1,0xE3,0xFE,0xF8,0xFC,0xFF,0xFF,0xFF,0xFC,
0xFC,0xFE,0xFF,0x70,0x50,0x50,0x58,0x48,0x4C,0x6E,0x6B,0x6D,0x24,0x44,0x42,0x42,
0x42,0x43,0x41,0xC1,0xC1,0xA0,0x20,0x20,0x30,0x10,0x18,0x08,0x04,0x03,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xE0,0x20,
0x10,0x18,0x08,0x04,0x04,0x04,0x03,0x83,0xC3,0x61,0x21,0x11,0x11,0x09,0x0D,0x04,
0x04,0x02,0x02,0x02,0x03,0x81,0xC1,0x61,0x33,0x97,0x7F,0x1F,0x1F,0x1F,0x3F,0xFF,
0x0F,0x0F,0x0F,0x16,0x24,0xCC,0xC8,0x88,0x10,0x10,0x10,0x30,0x20,0x20,0x20,0x20,
0x60,0x40,0x40,0x40,0x40,0xC1,0xC1,0x41,0x42,0x62,0x24,0x2C,0x10,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,
0x01,0x03,0x02,0x32,0x2E,0x42,0x61,0x20,0x20,0x20,0x20,0x20,0x90,0x70,0x18,0x08,
0x08,0x04,0x04,0x02,0x01,0x80,0xE0,0x18,0x06,0x01,0x00,0x00,0x00,0x00,0x00,0xFF,
0x3C,0xE0,0x80,0x00,0x00,0x00,0x01,0x03,0x0D,0x3B,0xE2,0x06,0x04,0x0C,0x08,0x10,
0x10,0x30,0xE0,0x20,0x20,0x20,0x20,0x13,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7E,0xC3,0x40,0xE0,0x60,
0x10,0x10,0x08,0x06,0x83,0xFF,0x00,0x00,0x00,0x00,0x00,0x80,0xE0,0xF8,0x0F,0x3F,
0xC0,0x00,0x01,0x06,0x18,0x30,0x60,0xC0,0x80,0x80,0x80,0xFF,0x0C,0x08,0x08,0x10,
0x10,0x10,0x10,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x03,
0x02,0x02,0x02,0x01,0x00,0x01,0x06,0x0C,0x04,0x04,0x02,0x01,0x00,0x1F,0x60,0x80,
0x81,0x67,0x1C,0x08,0x10,0x10,0x08,0x07,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D2[1024]=	   // Eagle
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0xF8,0x80,0x00,0xFF,0xFF,
0x80,0xE0,0xFE,0xBF,0xE2,0x60,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x07,0x3F,0xFE,0xFE,0xFF,0xF8,0xFC,0xE0,0xE0,0xC0,0xC0,0x80,
0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,0xF9,0xFC,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x0F,0x1F,0x1F,0x3F,0x7F,0xFF,0xFF,0xFF,
0xFF,0xFC,0xFC,0xFC,0xE0,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0xE0,0xF8,0xFE,0xFC,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xDF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x1F,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xFC,0xC0,0x00,0x00,0x00,0x80,0xC0,0xE0,0xE0,
0xF0,0xF0,0xF8,0xFC,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x0F,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0xF0,0xF8,0xFC,0xFC,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0x7F,0x3F,0x1F,0x0F,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1F,0x1F,0x07,0x03,0x03,
0x07,0x87,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xC7,0x87,0x8F,0x0F,0x07,0x07,0x07,0x03,0x01,0x01,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x30,0x30,0xF0,0x70,0x70,0xF0,0xFE,
0x3F,0x3F,0x3F,0x1F,0x0F,0xEF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xFE,0xFC,0xFC,0xF8,0xF8,0xF0,0xE0,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x63,0x79,
0x1C,0x9C,0xFC,0x3C,0x1E,0x1F,0x07,0x07,0x07,0x07,0x03,0x01,0x03,0x07,0x03,0x07,
0x0F,0x0F,0x0F,0x1F,0x0F,0x0F,0x0F,0x0F,0x07,0x07,0x07,0x07,0x07,0x03,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D3[1024]=	   // butterfly1
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,0xFE,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xFE,0xFC,0xFC,0xF8,0xF8,0xF0,0xE0,
0xC0,0xC0,0x80,0x03,0x1E,0x78,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x80,0xE0,0x38,0x1F,0x03,0x80,0xC0,0xE0,0xE0,0xF0,0xF8,
0xF8,0xFC,0xFC,0xFE,0xFE,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x07,0x3F,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xBF,0xA5,0x81,0x01,0x00,
0x01,0x03,0x07,0x0F,0x1E,0x7C,0x79,0xE3,0xCE,0x30,0xC0,0x80,0x00,0x00,0x00,0x00,
0x00,0x80,0xE0,0x38,0xCE,0xF7,0xF9,0x3C,0x1E,0x0F,0x07,0x03,0x01,0x00,0x01,0x83,
0x87,0x9F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x1F,0x03,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x0F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xB7,0xB7,0xB6,0x36,0x30,0x30,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x03,0x0F,0x3C,0x73,0xEE,0xFC,0xF8,0xF8,
0xC6,0x71,0x3C,0x0F,0x07,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x30,0xB6,
0xB6,0xB7,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x0F,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x01,0x07,0x0F,0x0F,0x1F,0xDF,0xFF,0xFF,0xFF,0xFD,0xF9,0xF8,0xF8,0xF8,0x78,
0x78,0x1C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0D,0x0D,0x1F,0x3F,0xFF,0xFF,0xFF,0xFF,
0xFF,0x1F,0x1F,0x0D,0x0D,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x78,0xF8,0xF8,0xF8,
0xF9,0xF9,0xFB,0xFF,0xFF,0xFF,0xDF,0x1F,0x0F,0x07,0x07,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0xDC,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xD3,0xE0,0x60,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xF8,0x3E,0x1F,0xFF,0xFF,0xFF,
0x07,0x7E,0xF0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xC3,0x93,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x01,0x03,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFE,0xF8,0xF8,0xC0,0x80,0xC0,0xE0,0x80,0xF0,0x3F,0x03,0x00,0x00,0x01,0x01,0x01,
0x00,0x00,0x07,0x7F,0xF0,0x80,0xE0,0xC0,0x80,0xC0,0xF8,0xF8,0xFE,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x67,0x03,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0xC3,0xFF,0xFF,0x7F,0x7F,0x7F,0x7F,0x7F,
0x7F,0x7F,0x7F,0x7F,0x7F,0x3F,0x1F,0x0F,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x0F,0x0F,0x1F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,0x7F,
0x7F,0x7F,0xFF,0xFF,0x81,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0xFE,0xFF,0x1F,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x01,0x7F,0xFF,0xFE,0x7C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D4[1024]=	   // Horse
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0x40,0xE0,0xE0,
0xF0,0xF0,0xE8,0xE8,0xFC,0xFE,0xFE,0xFC,0xFE,0xF0,0xE6,0xCE,0xBE,0x7C,0xFC,0xF8,
0xF8,0xD8,0x10,0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x09,0x0F,0x07,
0x07,0x87,0xC3,0xE7,0xF7,0xF7,0xFB,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x18,0xC3,0xFF,
0xFF,0xFF,0x69,0x29,0x29,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x80,0xC0,0xE0,0x70,0x38,0x1E,0x1E,0x3E,0x78,0xF0,0xE0,0xC0,0xC0,0xE0,0xF8,0xFC,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xE3,0xF8,0xBF,0x1F,
0x0F,0x0A,0x02,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0xE0,0xF0,0xF8,0xF0,0xF8,0xFC,0xFC,0xFC,0xFC,0xD8,0xD8,0xD8,0xD8,0x9C,
0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,
0x3F,0x3F,0x37,0x03,0x00,0x00,0xF8,0xF8,0x38,0x3B,0x33,0x71,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0x7F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFE,0xFC,0xFC,0xF8,0xF8,0xF0,0xF0,0xF0,0xF0,0xE0,0xE0,0xE0,0xC0,0x80,0x00,0x00,
0x00,0x00,0x03,0x9F,0xFF,0xFF,0xFF,0xFF,0xDF,0xDF,0xDF,0x0F,0x0E,0x0A,0x02,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x0F,0x1E,0x3E,0xF8,0xE0,0xE0,0x00,0x01,0x01,
0x01,0x01,0x00,0x03,0x07,0x07,0x0F,0x0F,0x1F,0x1F,0x3F,0x3F,0x3F,0x7F,0x7F,0x7F,
0x7F,0x7F,0x7F,0x7F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFE,0x0E,0x0F,0x1F,0x3F,0x1F,0x3F,0x07,0x0F,0x03,0x02,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x06,0x0E,0x1E,0xFC,0xFB,0xFF,0xFF,0x7F,0x7F,0xFF,0xFF,0xFF,0xFF,0xFF,0xEF,0xC3,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,
0xC0,0xF0,0xF8,0x1F,0x0F,0x07,0x03,0x01,0x00,0x00,0x00,0x01,0x83,0xE7,0xFF,0x3F,
0x0F,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x0F,0x0F,0x05,
0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x78,0x78,0x7C,0x2E,0x07,0x03,0x03,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D5[1024]=	   // Oak tree
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,0x60,0x60,0x30,0x20,
0x30,0x60,0x60,0xC0,0xC0,0x40,0x48,0xFC,0xB6,0x01,0x01,0x01,0x03,0x02,0x04,0x04,
0x02,0x03,0x0E,0x38,0x60,0xC0,0x60,0x30,0x30,0x60,0x20,0x70,0xC0,0x80,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x0B,0x3F,0x60,0xC0,0x80,0x80,0x80,
0x00,0x80,0x00,0x01,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xC0,0x60,0x20,0x96,0x9D,0x00,
0x80,0x00,0x00,0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0xC0,0xF0,0x18,0x08,0x04,
0x04,0x03,0x01,0x01,0x01,0x01,0x03,0x02,0x06,0x06,0x06,0x0C,0x18,0x19,0x0D,0x05,
0x07,0x03,0x00,0x00,0x00,0x80,0x80,0xC0,0xC0,0x00,0x00,0x00,0xC0,0xE0,0x38,0x08,
0x04,0x04,0x04,0x0C,0x04,0x04,0x06,0x06,0x06,0x0B,0x09,0x0C,0x07,0x01,0x01,0x01,
0x03,0x02,0x01,0x01,0x07,0xED,0x98,0x80,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x7F,0xC7,0x00,0x00,
0x80,0xC0,0x80,0x00,0x00,0x00,0xC0,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x03,
0x07,0x86,0x8C,0x78,0x30,0x30,0x58,0x9E,0xE1,0x83,0x27,0x19,0x87,0x4C,0x98,0x30,
0x30,0x20,0x20,0xE0,0x20,0x20,0x20,0x00,0x00,0xA0,0x80,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x60,0xE0,0xE0,0xA0,0xB0,0xD0,0x69,0x27,0x30,0x20,0x20,0xE0,0xC0,0xC0,
0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,
0xC0,0x41,0x43,0x62,0x63,0x3B,0x08,0x1B,0x32,0x33,0x09,0x07,0x01,0x00,0x20,0x94,
0x4C,0x20,0x10,0x00,0x00,0x08,0x0C,0x0C,0x78,0x44,0x25,0x22,0x11,0x30,0xEF,0x00,
0x00,0x00,0x04,0x80,0xC6,0x46,0x5A,0x4A,0x4D,0x44,0x44,0x48,0x17,0x0D,0x10,0x30,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC7,
0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x0F,0x18,0x18,0x10,0x70,0xC0,0xC0,0x80,0x00,0x00,0x80,0x80,0xC0,0x60,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x40,0xC0,0xA0,0x20,0x30,0x10,0xF8,0x08,0x07,0x03,
0x00,0xC1,0x30,0x61,0x60,0x60,0x30,0xB0,0xF0,0x20,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x80,0x80,0x1C,0x1C,0x34,0xE4,0x26,0x04,0x03,0x01,0x01,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x01,0x00,0x00,0x0F,0x09,0x08,
0x08,0x08,0x08,0x0C,0x0C,0x08,0x06,0x06,0x03,0x00,0x00,0x00,0xFF,0x00,0x00,0x00,
0x01,0xFF,0x00,0x00,0x00,0x00,0x00,0x01,0x06,0x04,0x04,0x0C,0x08,0x18,0x18,0x08,
0x04,0x04,0x06,0x03,0x01,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0x63,0x1E,0x00,0x00,
0x00,0x1F,0x60,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D6[1024]=	   // leaf
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x06,0x06,0x0E,0x3E,
0xFE,0xFC,0xFC,0xEC,0xDC,0xB8,0x78,0xF0,0xE0,0xE0,0xC0,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x03,0x0F,0x3F,0x7F,0xFF,0xFE,0xFC,0xE3,0x8F,0x3F,0xFF,0xFE,0xF8,0x80,0x00,
0x00,0x00,0x00,0x00,0xC0,0xE0,0xE0,0xF0,0xF8,0xF8,0x7C,0x3E,0x8E,0xE7,0xF9,0x0F,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x80,0xC0,0xC0,0xC0,0xE0,0xE0,0xE0,0xF0,0xF0,0xF0,0xF8,0xF8,0xF8,0xF8,0xF8,
0xF8,0xF8,0xF8,0xF0,0xF0,0xE0,0xE1,0xC3,0x83,0x03,0x03,0x03,0xF7,0x83,0x00,0x00,
0x00,0xF0,0xFE,0xFF,0xFF,0x9F,0xCF,0xE7,0xF3,0xF8,0x7E,0x3F,0x0F,0x03,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x01,0x01,0x01,0x01,0x03,0x03,0x03,0x07,0x07,0x07,0x0E,0x0E,0x0E,0x0E,
0x0E,0x0E,0x1F,0x1F,0x1E,0x0E,0x0D,0x0D,0x0F,0x00,0x00,0x00,0x00,0x0F,0xF8,0x00,
0x07,0x0F,0x0F,0x0F,0x87,0x87,0xC7,0xC3,0xC1,0xC0,0xC0,0x80,0x80,0x80,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xE0,0xF0,
0xF8,0xF8,0xFC,0xFC,0xFE,0xFE,0x7E,0x7E,0x3E,0x9E,0xCE,0xC4,0x00,0x00,0x07,0xFF,
0x00,0x00,0x02,0x77,0xF7,0xF7,0xF7,0xF7,0xF7,0xF7,0xF7,0xF7,0xF7,0x77,0x77,0x77,
0x77,0x66,0x6C,0x6C,0x68,0x50,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0xF0,0xF8,0xBC,0xDF,0xCF,0xEF,0x7F,
0x77,0x7B,0x7B,0x3D,0x3D,0x1E,0x1E,0x0F,0x0F,0x07,0x01,0x00,0x00,0x00,0x00,0xFF,
0xFF,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,
0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0x70,0x7F,
0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D7[1024]=	   // dragon
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x1C,0x08,
0x48,0xC8,0xEC,0xFC,0xFC,0xFC,0xFC,0x7C,0x7E,0x4E,0x9E,0x9F,0x1F,0x16,0x16,0x36,
0x36,0xE6,0xE6,0x52,0xF2,0xF2,0xF2,0x90,0x90,0x10,0x10,0x10,0x20,0x20,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0xC0,0xC0,0xE0,0xE0,0x78,0x70,0xF0,0xF8,0xF8,0xF0,0xF0,0xF8,
0xF0,0x60,0xF0,0xF1,0x30,0xE0,0xC1,0xE1,0xF0,0x78,0xBC,0xDC,0xEC,0xEE,0x36,0x07,
0x63,0x71,0x70,0xF1,0xF1,0xF3,0xC2,0xC0,0x81,0x01,0x02,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x01,0x03,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x03,0x06,0x00,0x01,
0x03,0x0E,0x38,0x00,0x03,0x1F,0x7F,0xC1,0x3C,0xFF,0xFF,0xFF,0xF0,0x61,0x40,0xC0,
0x80,0x00,0x00,0x03,0x07,0x0F,0x1F,0x17,0x12,0x14,0x24,0x7C,0xB8,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x01,0xFB,0x77,0xE7,0xEE,0xCE,
0xDC,0x9C,0x18,0xB8,0x38,0x48,0x88,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x10,0x10,0x90,0xD0,0x58,0x78,0xFC,0x7C,0x0E,0x21,0x1F,0x01,
0x03,0xFF,0x1F,0x06,0x0D,0x02,0x0C,0xF8,0xE2,0x17,0x37,0x77,0xEF,0x2F,0x5F,0x14,
0x24,0x48,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x02,0x02,0x01,0x01,0x01,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x80,0x80,
0x80,0xC0,0xE0,0x60,0x60,0x30,0x1C,0x0F,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x10,0x08,0x78,0x0E,0x76,0xFA,0x07,0x07,0x03,0x01,0x01,0x01,0x81,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x02,0x07,0x4E,0x4C,0x5C,0x79,
0xE6,0x1C,0x78,0xF0,0xE4,0xF8,0xF0,0xF0,0xE0,0xC0,0xC0,0x80,0x80,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D8[1024]=	   // baby boy
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x80,0xC0,0x40,0x20,0x30,0x10,0x10,0x08,0x08,0x08,0x18,0x19,0x1F,0x0F,0x0A,
0x10,0x10,0x20,0x20,0x40,0x80,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x38,0x0C,
0x03,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x20,0xA0,0x80,0x00,0x80,0x20,0x40,
0x40,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x0C,0x30,0xC0,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0C,0x1F,0xFF,0x80,0x00,0x00,
0x80,0x80,0x80,0xC0,0x60,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x01,0x01,0x01,0x00,
0x00,0x00,0x20,0x60,0xC0,0x40,0x40,0x40,0x00,0x00,0xF7,0x7F,0x3E,0x0C,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x18,0x60,
0xC0,0x00,0x00,0x0F,0x38,0x60,0x40,0x80,0x80,0x80,0x00,0x00,0x00,0x00,0x80,0x80,
0xC0,0x40,0x30,0x1C,0x07,0x00,0x80,0xC0,0x70,0x1C,0x07,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0x20,0x10,0x08,0x04,
0x06,0x03,0x02,0x84,0x0C,0x08,0x18,0x10,0x10,0x11,0x11,0x31,0x31,0x11,0x10,0x10,
0x10,0x08,0x08,0x04,0x06,0x83,0x01,0x02,0x02,0x04,0x08,0x10,0x60,0xC0,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x06,0x3C,0xC8,0x88,0x10,
0xF0,0x08,0x07,0x07,0x02,0x02,0x92,0x9A,0x9C,0xE0,0x00,0x00,0x00,0x00,0x60,0x94,
0x9E,0x92,0x12,0x02,0x02,0x07,0x04,0x18,0xF0,0x18,0x88,0x6C,0x3C,0x06,0x03,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0xF0,0x98,0x04,0x02,0xC3,0xE1,0x11,0x39,0x05,0x0F,0x09,0x01,0x03,0x07,
0x0F,0x1C,0x78,0xD0,0x10,0x10,0x10,0x14,0x1C,0x0B,0x08,0x08,0x08,0x08,0x0B,0x0E,
0x16,0x12,0x10,0x90,0xD0,0x70,0x18,0x0E,0x07,0x03,0x01,0x09,0x07,0x25,0x3A,0x92,
0xE2,0x84,0x0C,0xF8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x01,0x02,0x07,0x0C,0x18,0x10,0x30,0x20,0x60,0x40,0x40,0xC0,0x80,
0x80,0x80,0x80,0x40,0x63,0x1E,0x08,0x08,0x18,0x10,0x10,0x10,0x10,0x10,0x10,0x08,
0x08,0x08,0x3E,0x63,0xC0,0x80,0x80,0x80,0x80,0xC0,0x40,0x40,0x60,0x20,0x30,0x10,
0x18,0x0C,0x07,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D9[1024]=	   // butterfly2
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF0,
0xF8,0xFF,0xFF,0xFC,0xFA,0xF4,0xF8,0xFC,0xFF,0xA3,0x00,0x40,0x80,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFE,0xFC,0xFC,0xF8,
0xF0,0xE0,0xE0,0xC0,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xC0,0xE3,0xFF,
0xDF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x1F,0x2F,0x07,0x07,0x07,0x07,0x07,
0x0F,0x0F,0x0F,0x3F,0x7F,0x7F,0xFE,0xF8,0xF0,0xC0,0x80,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0xC0,0x70,0x7C,0x8E,0xD6,0xC7,0xE0,0x60,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0F,0x3F,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x3F,0x3C,0x08,0x18,0x1C,0x08,0x18,0x10,
0x38,0x30,0x30,0xE0,0xC0,0x80,0x43,0x0F,0x7F,0xFF,0xFF,0xF8,0xE0,0x00,0xC0,0x00,
0x60,0xFC,0xDE,0xFE,0x3F,0x07,0x01,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xD1,0xFB,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFC,0xF8,0xFA,0xF8,0xF8,0xF8,0xF8,0x78,
0xF8,0xF8,0xF8,0xFA,0xF8,0xFB,0xF7,0xE6,0xF8,0x9F,0xFF,0xEF,0xFF,0xFC,0x7E,0x3F,
0x0F,0x17,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x80,0xA0,0xF0,0xF8,0xFC,0xFE,0xFE,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xF0,0xE0,0xC0,0xE0,
0xE0,0xE0,0xE0,0xF1,0xF1,0xF9,0xFD,0xFE,0xFF,0xBF,0x3F,0x0F,0x01,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x3F,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
0xFF,0xFF,0x7F,0x3F,0x3F,0x1F,0x1F,0x1F,0x0F,0x0F,0x0F,0x0F,0x0F,0x07,0x7F,0xFF,
0xFF,0x7F,0x3F,0x1F,0x0F,0x0F,0x07,0x03,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x01,0x03,0x07,0x1F,0x3F,0x3F,0x7F,0xFF,0xFF,0xFF,0xDF,
0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};

INT8U graphic_D10[1024]=	   // no leaves tree
{
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x80,0x00,0x60,0xF0,0x80,0x00,
0x00,0x00,0x00,0x00,0x00,0x04,0xF0,0xE0,0x78,0x1E,0x06,0x38,0x70,0xE0,0x80,0xF1,
0x7E,0x16,0x18,0x1A,0xCC,0x64,0x80,0x80,0x00,0x00,0x00,0x00,0x20,0xE0,0xF0,0x00,
0x00,0x00,0x00,0x80,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,
0xE0,0x80,0x00,0x00,0x44,0x4C,0x98,0xB8,0xC0,0x03,0x07,0x0F,0x18,0xFF,0xF7,0x00,
0x08,0x18,0x18,0x38,0x60,0xE0,0xFA,0xFB,0x0C,0x38,0x78,0xE0,0x7F,0x07,0x37,0xFF,
0x76,0x86,0xC2,0x73,0x33,0x01,0x01,0x88,0x38,0x00,0x00,0x00,0xE0,0xBC,0x01,0x00,
0x00,0xFF,0xFC,0x1F,0x01,0x00,0x00,0xE0,0xC0,0x80,0x40,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x80,
0x01,0x03,0x07,0x04,0x0C,0x98,0x58,0x73,0x7F,0x7E,0xF0,0xC0,0x8C,0x9F,0xFF,0xFC,
0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xFD,0xE0,0xF8,0xFF,0x61,0x03,0x07,0x1E,0xFF,
0xFF,0x81,0xC0,0xC0,0x60,0x70,0x38,0xFE,0xFE,0xC8,0x30,0x3E,0x0F,0x00,0x00,0x00,
0xF0,0xFF,0x8F,0xC1,0xE0,0x70,0x38,0x3F,0x31,0x10,0x00,0x00,0xF0,0x00,0xE0,0x78,
0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x60,0x62,0x66,0xC0,0x85,0x07,
0x0C,0x08,0x18,0x30,0x60,0xC0,0x80,0x00,0x00,0x00,0x10,0x70,0x31,0x33,0x37,0x3F,
0x3E,0x78,0xE0,0xC0,0x80,0x00,0xFB,0xFF,0xFB,0x07,0x1F,0xE0,0x80,0x00,0x00,0xFB,
0xFF,0xFF,0xE1,0x70,0x38,0x0C,0x1F,0xE3,0xF1,0x1C,0x00,0xC0,0xC0,0x70,0x38,0x1F,
0x87,0xC3,0xC3,0x21,0xA0,0x20,0x00,0x00,0x00,0x00,0x00,0x80,0xFF,0x6F,0x31,0x18,
0x0C,0x04,0x00,0x80,0xC0,0x00,0x00,0x80,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x06,0x06,0x02,0x40,0x43,0x47,
0x46,0x6C,0x68,0x18,0x70,0x71,0x67,0x6E,0x78,0xF0,0xE0,0xC0,0xBC,0xF0,0x84,0x4C,
0x78,0x70,0x70,0x21,0x63,0xC7,0xDF,0xFF,0xFD,0xF0,0x80,0x00,0x01,0x03,0x06,0xFF,
0xFF,0x2F,0x01,0x00,0x00,0x00,0xC0,0xFF,0xFD,0x38,0x1F,0x07,0x07,0x03,0x03,0x01,
0xE1,0xF0,0x00,0x40,0x00,0x00,0x00,0x80,0xC0,0xE0,0xFC,0xCE,0xC2,0x60,0x60,0x70,
0x70,0x78,0x0E,0x07,0x04,0x02,0x03,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0A,0x0A,0x0E,0x8C,0xCC,0xCC,0xC8,0x44,
0x5C,0x58,0xD8,0xF0,0x30,0x72,0x64,0x6C,0x78,0x70,0xE6,0xE6,0xC2,0xC7,0xC7,0xC6,
0xDC,0xF8,0xF8,0xD0,0xE0,0xE0,0xC0,0xC0,0x81,0x8F,0x8F,0xBC,0x70,0xC0,0xFC,0x3F,
0xFF,0xE0,0xE0,0x78,0xBC,0x9F,0x87,0xC3,0xC0,0xC0,0xC0,0xC0,0xC0,0xF0,0xFC,0xE3,
0xFB,0xFA,0xDE,0x8E,0x86,0x87,0xC3,0xC1,0xE1,0xF1,0xD0,0xC0,0xC0,0xC0,0xE0,0xE0,
0xF0,0xB8,0xBC,0x9E,0x1B,0x0C,0x04,0x02,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x0D,0x0C,0x06,0x06,0x06,0x02,0x03,0x03,0x01,0x01,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x01,0x03,0x06,0x06,0x7D,0xF3,0x00,
0xEF,0xFF,0xFF,0x0F,0x03,0x03,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
0x00,0x01,0x01,0x00,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x03,0x02,0x04,0x06,
0x00,0x01,0x01,0x00,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0xC0,0x80,0x40,0x70,0x70,0x20,0x60,0x60,0x3E,0x1F,0x00,
0x07,0x1F,0x1E,0x30,0x30,0x70,0x60,0x60,0x60,0xC0,0xC0,0x80,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
};